-- Pending binding update.
AS_BindingUpdatePending = nil;

---------------------------------------------------------------------------
-- result = AS_GetCurrentBindingSet()
--  Returns a table containing the current set of key bindings.  The table
--  is a mapping from key to command.
--
-- result - the binding set.
---------------------------------------------------------------------------
function AS_GetCurrentBindingSet()
  local numBindings = GetNumBindings();
  local result = {};
  for slot = 1,numBindings do
    local binding = {GetBinding(slot)};
    local command = binding[1];
    for i=2,getn(binding) do
      result[binding[i]] = command;
    end
  end
  return result;
end

---------------------------------------------------------------------------
-- AS_SetCurrentBindingSet()
--  Attempts to set the current bindings to match the given binding set.
---------------------------------------------------------------------------
function AS_SetCurrentBindingSet(bindingSet)
  AS_BindingUpdatePending = bindingSet;
  AS_UpdateBindingsNow();
end
  
---------------------------------------------------------------------------
-- result = AS_UpdateBindingsNow()
--  Attempts to update bindings now to match an enqueued binding set.
--
-- result - true if the update succeeded, false if the binding update has
--   not yet completed.
---------------------------------------------------------------------------
function AS_UpdateBindingsNow()
  -- No update pending?  Don't do anything.
  if not AS_BindingUpdatePending then
    return true;
  end
  
  -- If we're in combat, we can't programmatically change the bindings.
  if InCombatLockdown() then
    return false;
  end

  -- Give the binding set a better name.
  local bindingSet = AS_BindingUpdatePending;
  
  for k,v in pairs(bindingSet) do
    if v == 0 then      -- Bindings explicitly cleared by this profile map to 0, NOT nil
      AS_Automatic = true;
      SetBinding(k,nil);
      AS_Automatic = false;
    else
      -- Check if this key is already bound to the action
      local keys = {GetBindingKey(v)};
      local is_bound = false;
      for i,p in pairs(keys) do
        if p == k then
          is_bound = true;
          break;
        end
      end
      
      -- Attempt to only set bindings when we need to (save some performance).
      if not is_bound then
        AS_Debug("Setting key " .. k .. " to " .. v);
        AS_Automatic = true;
        SetBinding(k,v);
        AS_Automatic = false;
      end
    end
  end
  
  -- Must save character bindings or next time we go into the bindings
  -- menu they get unset again.
  SaveBindings(GetCurrentBindingSet());
  
  -- Done with the binding update.
  AS_BindingUpdatePending = nil;
  return true;
end

---------------------------------------------------------------------------
-- result = AS_VerifyBindingSet(bindingSet)
--   Verifies that the current set of key bindings match the given binding
--   set.
--
-- bindingSet - the binding set to test against the current key bindings.
-- result - true if they match, false if not.
---------------------------------------------------------------------------
function AS_VerifyBindingSet(bindingSet)
  for k,v in pairs(bindingSet) do
    if v ~= 0 then
      if GetBindingAction(k) ~= v then
        AS_Debug("ActionSwap verification failed because of key: " .. k);
        AS_Debug("v=" .. v .. ", GetBindingAction()=" .. GetBindingAction(k));
        return false;
      end
    end
  end
  return true;
end
