
-- List of action updates pending.
AS_ActionUpdatesPending = {};

---------------------------------------------------------------------------
-- action = AS_GetAction(slot)
--  Returns a table containing information about the specified action
--  button on the action bar.
--
-- slot - the slot on the action bar to retrieve information about.
-- action - a table containing information about the action, or nil if
--   there is no action in the slot.
---------------------------------------------------------------------------
function AS_GetAction(slot)
  assert(slot);
  local _type, id, subtype = GetActionInfo(slot);
  
  -- No action in the slot or invalid ID?
  -- (invalid ID actually occurs when ranking up a spell,
  -- so when we get the LEARNED_SPELL_IN_TAB event we refresh
  -- all actions)
  if not _type or id == 0 then
    return nil;
  end
  
  local action = {};
  action._type = _type;
  action.id = id;
  action.subtype = subtype;
  action.texture = GetActionTexture(slot);
  
  if _type == "spell" then
    local spellName, spellRank = GetSpellName(id, BOOKTYPE_SPELL);
    action.name = spellName;
    action.rank = spellRank;
    
  elseif _type == "item" then
    local itemName, itemLink = GetItemInfo(id);
    action.name = itemName;
    action.link = itemLink;
    
  elseif _type == "macro" then
    local macroName = GetMacroInfo(id);
    action.name = macroName;
    
  else
    AS_Print(AS_ERROR_UNKNOWN_ACTION_TYPE_1 .. _type .. AS_ERROR_UNKNOWN_ACTION_TYPE_2);
  end
  
  return action;
end

---------------------------------------------------------------------------
-- result = AS_SetAction(slot, action)
--  Immediately sets the action in an action bar slot.
-- 
-- slot - the slot to set the action of.
-- action - the action to put in the slot, or nil if actions are to
--   be removed from the slot.
-- result - true if the update succeeded, false if after setting the
--   action, it still isn't correct for some reason (probably because of
--   too many updates at once) and needs to be re-attempted.  "noexist"
--   is returned if the item or spell doesn't exist.
---------------------------------------------------------------------------
function AS_SetAction(slot, action)
  assert(slot, action);
  
  -- First, verify that the action actually needs to be changed.
  local curAction = AS_GetAction(slot);
  if AS_ActionsEqual(curAction, action) then
    return true;
  end
  
  -- If we're in combat, we can't programmatically change the actions.
  if InCombatLockdown() then
    return false;
  end
  
  -- Clear the cursor so we can grab things.
  ClearCursor();
  
  -- If we're trying to remove an action from a slot, that's easy.
  if not action then
    AS_Automatic = true;
    PickupAction(slot);
    ClearCursor();
    AS_Automatic = false;
    
  -- Are we trying to place a spell in the slot?  We have to first
  -- look it up in the spellbook by name and rank.
  elseif action._type == "spell" then
    local id = AS_LocateSpell(action.name, action.rank, action.id, true);
    if not id then -- Spell doesn't exist?  That makes things easy.
      AS_Debug("Warning: Spell " .. action.name .. " (" .. action.rank .. ") doesn't exist.");
      return "noexist";
    else
      AS_Automatic = true;
      PickupSpell(id, BOOKTYPE_SPELL);
      PlaceAction(slot);
      AS_Automatic = false;
    end
    
  -- Are we trying to place an item in the slot?  We have to look it
  -- up in our inventory and bags by item ID.
  elseif action._type == "item" then
    local invID, bagID, bagSlot = AS_LocateItem(action.id);
    if invID then
      AS_Automatic = true;
      PickupInventoryItem(invID);
      PlaceAction(slot);
      AS_Automatic = false;
    elseif bagID then
      AS_Automatic = true;
      PickupContainerItem(bagID, bagSlot);
      PlaceAction(slot);
      AS_Automatic = false;
    else
      AS_Debug("Warning: Item " .. action.link .. " doesn't exist.");
      return "noexist";
    end
  
  -- Are we trying to place a macro in the slot?  Look it up by name
  -- and texture - that's the best we can do since there's no unique
  -- identifier for a macro, and their indices might've changed since
  -- the profile was last used.
  elseif action._type == "macro" then
    local macroID = AS_LocateMacro(action.name, action.texture);
    if macroID then
      AS_Automatic = true;
      PickupMacro(macroID);
      PlaceAction(slot);
      AS_Automatic = false;
    else
      AS_Debug("Warning: Macro " .. action.name .. " (" .. action.texture .. ") doesn't exist.");
      return "noexist";
    end
    
  else
    AS_Debug("Unknown action type " .. action._type);
    return "noexist";
  end
  
  -- Bugfix: If it fails to place an action in its slot, it'll still be
  -- on the cursor, so make sure the cursor is now clear.
  ClearCursor();
  
  -- Now, verify that the action in the slot matches the action we intended
  -- to set it to.  If not, then return false indicating we must try again.
  return AS_ActionsEqual(AS_GetAction(slot), action);
end

---------------------------------------------------------------------------
-- result = AS_ActionsEqual(a1, a2)
--  Determines whether two actions are equivalent.
--
-- a1 - the first action to compare.
-- a2 - the second action to compare.
-- notExistIsNoAction - determines whether or not an item action with 0 of
--   that item in the player's inventory is equivalent to no action.
-- result - true if they are equivalent, false otherwise.
---------------------------------------------------------------------------
function AS_ActionsEqual(a1, a2, notExistIsNoAction)

  -- Is no item is same as no action?
  if notExistIsNoAction and not AS_ActionExists(a1) then
    a1 = nil;
  end
  
  if notExistIsNoAction and not AS_ActionExists(a2) then
    a2 = nil;
  end
  
  -- No action equals no action.
  if not a1 and not a2 then
    return true;
  end
  
  -- No action is not equal to an action.
  if (not a1 and a2) or (not a2 and a1) then
    return false;
  end
  
  -- If the types aren't equal, then they're not equal.
  if a1._type ~= a2._type then
    return false;
  end
  
  -- Spell compare, use name and rank.
  if a1._type == "spell" then
    return a1.name == a2.name and a1.rank == a2.rank;
  
  -- Item compare, use item ID.
  elseif a1._type == "item" then
    return a1.id == a2.id;
  
  -- Macro compare, use name and texture.
  elseif a1._type == "macro" then
    return a1.name == a2.name and a1.texture == a2.texture;
  end
  
  -- Don't know what type of action it is, so not equal.
  return false;
  
end

---------------------------------------------------------------------------
-- actionSet = AS_GetCurrentActionSet()
--  Returns the current set of actions on the action bar.  This set is
--  a table mapping slot number to action and does not include empty
--  slots.
--
-- actionSet - the resulting action set.
---------------------------------------------------------------------------
function AS_GetCurrentActionSet()
  local result = {};
  for slot = 1,120 do
    local action = AS_GetAction(slot);
    if action then
      result[slot] = action;
    end
  end
  return result;
end

---------------------------------------------------------------------------
-- AS_SetCurrentActionSet(actionSet)
--  Sets the actions on the action bar to match the given action set.  This
--  update may or may not occur immediately (some updates may be queued).
---------------------------------------------------------------------------
function AS_SetCurrentActionSet(actionSet)
  assert(actionSet);
  
  -- Update each slot ONLY IF it's not already up-to-date.
  for slot = 1,120 do
    local currentAction = AS_GetAction(slot);
    local newAction = actionSet[slot];
    if not AS_ActionsEqual(currentAction, newAction) then
      AS_RefreshAction(slot, actionSet);
    end
  end
end

---------------------------------------------------------------------------
-- AS_UpdateAction(slot, action)
--  Updates an action in the current profile.  If the action cannot be
--  found, it is removed from the profile and the action slot is emptied.
--  Note: The actual update is queued and may not be immediate.
--------------------------------------------------------------------------
function AS_UpdateAction(slot, action)
  assert(slot and action);
  
  -- If in the custom profile, just enqueue the update directly.
  if AS_CurrentProfile == AS_PROFILE_CUSTOM then
    local update = {["slot"]=slot, ["action"]=action};
    tinsert(AS_ActionUpdatesPending, update);
    AS_UpdateActionsNow();
    
  -- If in a profile, first modify the profile, then
  -- refresh the action bar.
  else
    local actionSet = AS_ProfileList[AS_CurrentProfile].actionSet;
    actionSet[slot] = action;
    AS_RefreshAction(slot);
  end
end

--------------------------------------------------------------------------
-- AS_RefreshAction(slot)
--  Refreshes the given slot on the action bar to match the data in the
--  current profile.  If it cannot, the profile is modified to remove the
--  faulty action and the action slot is emptied.
--  Note: The actual update is queued and may not be immediate.
--------------------------------------------------------------------------
function AS_RefreshAction(slot)
  assert(slot);
  
  -- If in the custom profile, we don't have to do anything.
  if AS_CurrentProfile == AS_PROFILE_CUSTOM then
    return;
  end
  
  -- Otherwise, enqueue an update.
  local actionSet = AS_ProfileList[AS_CurrentProfile].actionSet;
  local update = {["slot"]=slot, ["actionSet"]=actionSet};
  tinsert(AS_ActionUpdatesPending, update);
  AS_UpdateActionsNow();
end

---------------------------------------------------------------------------
-- result = AS_UpdateActionsNow()
--  Attempts to update as many actions as possible immediately, but when
--  an update fails, it stops, waiting a while before starting again.
--
-- result - true if all updates were successfully made, false if a delayed
--   update is pending.
---------------------------------------------------------------------------
function AS_UpdateActionsNow()
  while getn(AS_ActionUpdatesPending) > 0 do
    -- (slight inefficiency here, I wanted to make updates occur FIFO,
    -- though it might not matter in practice)
    local update = AS_ActionUpdatesPending[1];
    
    -- Get the action we're setting.
    local action;
    if update.actionSet then
      -- For normal profiles.
      action = update.actionSet[update.slot];
    else
      -- For custom profile only.
      action = update.action;
    end
    
    -- If this action is a spell, replace it with the
    -- closest match we can find.  Update the profile
    -- also.  (this is safety code in case a spell can't
    -- be found)
    if action and action._type == "spell" then
      action.id = AS_LocateSpell(action.name, action.rank, action.id, false);
      if not action.id then
        -- Couldn't find it.
        AS_Debug("Spell not found: " .. action.name .. " (" .. action.rank .. ").  Removing from action bar.");
        action = nil;
        if update.actionSet then
          update.actionSet[update.slot] = nil;
        end
      else
        -- Found it, but not the right rank.
        local spellName, spellRank = GetSpellName(action.id, BOOKTYPE_SPELL);
        if spellName ~= action.name or spellRank ~= action.rank then
          AS_Debug("Spell not found: " .. action.name .. " (" .. action.rank .. ").  Upranking to " .. spellName .. " (" .. spellRank .. ")");
          action.name = spellName;
          action.rank = spellRank;
          -- (update.actionSet is implicitly updated)
        end
      end
    end
    
    -- Place the action on the action bar.
    local result = AS_SetAction(update.slot, action);
    
    if result == "noexist" then
      -- The thing that is supposed to fill this slot wasn't
      -- found.  Update the profile and remove the action from
      -- this slot.
      update.actionSet[update.slot] = nil;
    elseif result then
      -- Success, remove from queue.
      tremove(AS_ActionUpdatesPending, 1);
    else
      -- Failed to update, try again later.
      return false;
    end
  end
  return true;  -- No more updates!
end

---------------------------------------------------------------------------
-- result = AS_VerifyActionSet(actionSet)
--   Verifies that the current action bar matches the given action set.
--
-- actionSet - the action set to test against the current action bar.
-- result - true if they match, false if not.
---------------------------------------------------------------------------
function AS_VerifyActionSet(actionSet)
  for slot = 1,120 do
    if not AS_ActionsEqual(AS_GetAction(slot), actionSet[slot], true) then
      AS_Debug("ActionSwap verification failed because of actions:");
      AS_DumpAction(AS_GetAction(slot));
      AS_DumpAction(actionSet[slot]);
      return false;
    end
  end
  return true;
end

---------------------------------------------------------------------------
-- Helper function to display an action.
---------------------------------------------------------------------------
function AS_DumpAction(action)
  if not action then
    AS_Debug("no action");
    return;
  end
  local msg = "type=" .. action._type .. ", id=" .. action.id;
  if action._type == "spell" then
    msg = msg .. ", spell name=" .. action.name .. ", spell rank=" .. action.rank;
  elseif action._type == "item" then
    msg = msg .. ", item link=" .. action.link .. ", item ID=" .. action.id;
  elseif action._type == "macro" then
    msg = msg .. ", macro name=" .. action.name .. ", texture=" .. action.texture;
  end
  AS_Debug(msg);
end

---------------------------------------------------------------------------
-- result = AS_ActionExists(action)
--   Helper function to determine whether or not the given action is
--   non-existant, that is, that it can't be selected from the user's
--   bags, spellbook, or macro pane. (i.e. if you have 0 of an item,
--   that item "doesn't exist", same thing with deleted macros, etc.)
--
-- action - the action to look for.
-- result - true if the action exists, false if not.
---------------------------------------------------------------------------
function AS_ActionExists(action)
  -- No action counts as not existing.
  if not action then
    return false;
  end
  
  -- Check if we have any of an item in our inventory.
  if action._type == "item" and GetItemCount(action.id) == 0 then
    return false;
  end
  
  -- Check if a spell doesn't exist in our spellbook.
  if action._type == "spell" and not AS_LocateSpell(action.name, action.rank, action.id, true) then
    return false;
  end
  
  -- Check if a macro can't be found.
  if action._type == "macro" and not AS_LocateMacro(action.name, action.texture) then
    return false;
  end
  
  return true;
end
