---------------------------------------------------------------------------
-- Supplemental Documentation
-- 
-- Fields in an action's table:
--  _type, id, subtype, texture, name, rank (for spells), link (for items)
-- Note: We need texture because macros are identified by name and texture.
-- 
-- Definitions:
--   Action set - a set of action buttons
--   Binding set - a set of key bindings
---------------------------------------------------------------------------

AS_CUR_CHAR_DATA_VERSION = 1;
AS_DEBUG_MODE = false;

-- Perform any pending updates at this interval.
AS_UPDATE_INTERVAL = 1.00; -- Will re-attempt any pending updates at this interval
AS_NextUpdate = 0.0;

AS_LoadEventOccurred = false;  -- Don't do anything until we're loaded...
AS_PostLoadOccurred = false;
AS_Loaded = false;

function AS_OnLoad()
  -- Register the localization.
  AS_RegisterLocalization();
  
  -- Register for events.
  this:RegisterEvent("ADDON_LOADED");
  this:RegisterEvent("ACTIONBAR_SLOT_CHANGED");
  this:RegisterEvent("UPDATE_BINDINGS");
  this:RegisterEvent("LEARNED_SPELL_IN_TAB");
  
  -- Set up the slash command.
  SlashCmdList["AS"] = AS_Command;
  SLASH_AS1 = "/as";
  SLASH_AS2 = "/actionswap";
end

function AS_OnLoadEvent()
  -- Set up the default data if none is found.
  if not AS_CharDataVersion then
    AS_CharDataVersion = AS_CUR_CHAR_DATA_VERSION;
    AS_ProfileList = {};
    AS_CurrentProfile = AS_PROFILE_CUSTOM;
  end
  
  AS_LoadEventOccurred = true;
end

function AS_OnPostLoad()
  -- Make sure it's on per-character bindings.
  if GetCurrentBindingSet() == 1 then
    AS_Error(AS_ERROR_ONLY_CHARACTERSPECIFIC);
    return;
  end
  
  -- Print the loaded message.
  AS_Print(AS_MSG_LOADED);
  AS_Loaded = true;
  
  -- Verify that the current action bar and bindings match
  -- the profile, otherwise switch to 'custom'.
  AS_Verify();
end

function AS_Verify()
  if AS_CurrentProfile ~= AS_PROFILE_CUSTOM then
    
    -- Remove any bindings from the current profile which conflict with
    -- bindings set by other mods.  Always do this before testing the
    -- profile against the current setup.
    AS_RemoveContradictingBindings(AS_CurrentProfile);
    
    -- Verify that the current action bar matches the action set of the
    -- current profile.  If it doesn't switch to the custom profile and
    -- warn the user.  This typically happens when using an old saved
    -- variables file after not having used the mod for a while. (i.e.
    -- the profiles are not up-to-date).
    if not AS_VerifyActionSet(AS_ProfileList[AS_CurrentProfile].actionSet) or not AS_VerifyBindingSet(AS_ProfileList[AS_CurrentProfile].bindingSet) then
      local old_profile = AS_CurrentProfile;
      
      -- Switch to the custom profile.
      AS_SwitchProfile(AS_PROFILE_CUSTOM);
      
      -- Display the warning dialog.
      AS_VerifyFailedDlg_Text:SetText(AS_WARN_VERIFY_FAILED_1 .. old_profile .. AS_WARN_VERIFY_FAILED_2);
      AS_VerifyFailedDlg:Show();
      
    else
      AS_Debug("Verification passed.");
    end
  else
    AS_Debug("Verification passed.");
  end
end

function AS_OnEvent()
  if event == "ADDON_LOADED" and arg1 == AS_ADDON_NAME then
    AS_OnLoadEvent();
  elseif event == "ACTIONBAR_SLOT_CHANGED" then
    AS_OnActionBarChanged(arg1);
  elseif event == "UPDATE_BINDINGS" then
    AS_OnBindingsChanged();
  elseif event == "LEARNED_SPELL_IN_TAB" then
    AS_OnLearnedNewSpell();
  end
end

function AS_OnUpdate(timePassed)
  -- Fire the loaded event?
  if not AS_PostLoadOccurred and AS_LoadEventOccurred then
    AS_PostLoadOccurred = true;
    AS_OnPostLoad();
  end
  
  if not AS_Loaded then
    return;
  end
  
  if timePassed > AS_NextUpdate then
    AS_UpdateActionsNow();
    AS_UpdateBindingsNow();
    AS_NextUpdate = AS_UPDATE_INTERVAL;
  else
    AS_NextUpdate = AS_NextUpdate - timePassed;
  end
end

---------------------------------------------------------------------------
-- AS_OnActionBarChanged(slot)
--  Called whenever anything on the action bar is changed.
---------------------------------------------------------------------------
function AS_OnActionBarChanged(slot)
  if not AS_Loaded then
    return;
  end
  
  -- If we're in the custom profile, we shouldn't do anything, and if
  -- this is an automatic change, we shouldn't do anything.
  if AS_Automatic or AS_CurrentProfile == AS_PROFILE_CUSTOM then
    return;
  end
  
  -- Update the action in the current profile.
  AS_ProfileList[AS_CurrentProfile].actionSet[slot] = AS_GetAction(slot);
end

---------------------------------------------------------------------------
-- AS_OnBindingsChanged()
--  Called whenever key bindings are changed.
---------------------------------------------------------------------------
function AS_OnBindingsChanged()
  if not AS_Loaded then
    return;
  end
  
  -- Don't let the user change to character specific bindings.
  if GetCurrentBindingSet() == 1 then
    AS_Error(AS_ERROR_ONLY_CHARACTERSPECIFIC);
    AS_Loaded = false;  -- unload the mod.
    return;
  end
  
  -- Don't need to do anything if we're in the custom profile or if
  -- this is an automatic change.
  if AS_Automatic or AS_CurrentProfile == AS_PROFILE_CUSTOM then
    return;
  end
  
  local profile = AS_ProfileList[AS_CurrentProfile];
  local bindings = AS_GetCurrentBindingSet();
  
  -- Update the bindings in the profile.
  for k,v in pairs(bindings) do
    profile.bindingSet[k] = v;
  end
  
  -- Any keys which exist in the profile but not in the current bindings
  -- need to be set to 0 to indicate they are unbound in this profile.
  for k,v in pairs(profile.bindingSet) do
    if not bindings[k] then
      profile.bindingSet[k] = 0;
    end
  end
end

---------------------------------------------------------------------------
-- AS_OnLearnedNewSpell()
--  Called after a new spell is learned and that spell is
--  ranked up on the action bars.
---------------------------------------------------------------------------
function AS_OnLearnedNewSpell()
  if not AS_Loaded then
    return;
  end
  
  -- We don't need to update anything if in the custom profile.
  if AS_CurrentProfile == AS_PROFILE_CUSTOM then
    return;
  end
  
  -- Since the upranked spell has an invalid ID on the action
  -- bar when we receive the action bar changed event, we must
  -- perform a full update of the profile's action bars to
  -- capture the change after the fact.
  AS_ProfileList[AS_CurrentProfile].actionSet = AS_GetCurrentActionSet();
end

---------------------------------------------------------------------------
-- AS_Command(msg)
--   Processes /as commands.
--   msg - arguments to the command.
---------------------------------------------------------------------------
function AS_Command(msg)
  if not AS_Loaded then
    return;
  end
  
  assert(msg);
  
  -- Split on spaces.
  local args = {};
  for arg in string.gmatch(msg, "[^ ]+") do
    table.insert(args, arg);
  end
  
  -- Make sure msg contains at least 1 argument.
  if table.getn(args) == 0 then
    for k,v in pairs(AS_SYNTAX) do AS_Print(v); end
    return;
  end
  
  -- We don't care about the case of the command.
  args[1] = args[1]:lower();
  
  -- Test command.
  if args[1] == AS_COMMAND_TEST then
    AS_Test();
  
  -- List profiles command.
  elseif args[1] == AS_COMMAND_PROFILES then
    AS_ListProfiles();
  
  -- Create profile command.
  elseif args[1] == AS_COMMAND_CREATE then
    if table.getn(args) ~= 2 then
      AS_Print(AS_COMMAND_CREATE_SYNTAX);
      return;
    end
    AS_CreateProfile(args[2]);
  
  -- Delete profile command.
  elseif args[1] == AS_COMMAND_DELETE then
    if table.getn(args) ~= 2 then
      AS_Print(AS_COMMAND_DELETE_SYNTAX);
      return;
    end
    AS_DeleteProfile(args[2]);
    
  -- Delete all profiles command
  elseif args[1] == AS_COMMAND_DELETEALL or args[1] == AS_COMMAND_DELETEALL_2 then
    AS_DeleteAllProfiles();
  
  -- Switch profile command.
  elseif args[1] == AS_COMMAND_PROFILE then
    if table.getn(args) ~= 2 then
      AS_ListProfiles();
      return;
    end
    AS_SwitchProfile(args[2]);
  
  -- Uprank spells command.
  elseif args[1] == AS_COMMAND_RAISE_SPELLS or args[1] == AS_COMMAND_RAISE_SPELLS_2 then
    AS_RaiseAllSpells();
    
  -- Replace profile command.
  elseif args[1] == AS_COMMAND_REPLACE then
    if table.getn(args) ~= 2 then
      AS_Print(AS_COMMAND_REPLACE_SYNTAX);
      return;
    end
    AS_ReplaceProfile(args[2]);
  
  -- Secret command: force verify now.
  elseif args[1] == "verifynow" then
    AS_Verify();
    
  else
    AS_Error(AS_ERROR_UNKNOWN_COMMAND_1 .. args[1] .. AS_ERROR_UNKNOWN_COMMAND_2);
  end

end
