-- ActionCombat - a combat addon for World of Warcraft
-- Description: Shows animations for in-combat events
-- Author: Simon Fuhrmann, Doomsday@Arguswacht DE

--[ Animaition config ]------------------------------------------------

--local textureAmount = 3 -- Number of textures for the animations (max 10)

local iconSize = 50 -- quasi-Pixel size of anis
local iconSpacing = 30 -- quasi-Pixel spacing between anis
local realignSpeed = 100 -- quasi-Pixels per second

local aniReactiveIdleLen = 2.0 -- Reactives are shown that long
local aniDeactivateLen = 0.3 -- Animation speed to deactivate icons
local aniFadeinLen = 0.15 -- Length of the fade in for new animations
local aniFadebackLen = 0.10 -- Length of the face back for new animations
local aniNoticeLen = 0.4 -- Time notices take until they disappear
local aniFadeInSizeFactor = 2 -- Factor the icon size will be multiplied with
local aniNoticeSizeFactor = 3 -- Size multiplication factor for notices

local aniBaseAlpha = 0.8 -- Alpha value for animations
local aniFadedBaseAlpha = 0.3 -- Alpha value for faded animations (OOC)

local windowDebugAlphas = { 0.0, 0.3 } -- Alpha values for the animation bars
local windowDebugStayOnChange = 3.0 -- Visible for this seconds at change

--[ Init Addon ]-------------------------------------------------------

ActionCombat = AceLibrary("AceAddon-2.0"):new("AceConsole-2.0",
    "AceEvent-2.0", "AceDB-2.0", "FuBarPlugin-2.0")
ActionCombat.hasIcon = true
ActionCombat.independentProfile = true
ActionCombat.hideWithoutStandby = true

local L = AceLibrary("AceLocale-2.2"):new("ActionCombat") -- String localization
local timeleft = 0 -- Current animation time (clock)
local inCombat = false
local inMeeleCombat = false

--[ Player class config ]----------------------------------------------

local locClass, engClass = UnitClass("player")
local isWarrior = (engClass == "WARRIOR")
local isWarlock = (engClass == "WARLOCK")
local isDruid = (engClass == "DRUID")
local isShaman = (engClass == "SHAMAN")
local isMage = (engClass == "MAGE")
local isHunter = (engClass == "HUNTER")
local isPaladin = (engClass == "PALADIN")
local isPriest = (engClass == "PRIEST")
local isRogues = (engClass == "ROGUES")
local allClasses = { "WARRIOR", "WARLOCK", "DRUID", "SHAMAN",
    "MAGE", "HUNTER", "PALADIN", "PRIEST", "ROGUES" }

--[ Event icons ]------------------------------------------------------

-- Warrior icons
local iconExecute = "Interface\\Icons\\INV_Sword_48"
local iconOverpower = "Interface\\Icons\\Ability_MeleeDamage"
local iconRevenge = "Interface\\Icons\\Ability_Warrior_Revenge"
local iconVictoryRush = "Interface\\Icons\\Ability_Warrior_Devastate"
-- Hunter icons
local iconKillCommand = "Interface\\Icons\\Ability_Hunter_KillCommand"
-- Notice icons
local iconKillingBlow = "Interface\\Icons\\Ability_Creature_Cursed_02"

--[ Selfbuff events ]--------------------------------------------------

local selfBuffEvents = {}
local selfBuffAniRequest = nil

function AC_insertSelfBuff (unlocName, class)
  table.insert(selfBuffEvents, {
      id = unlocName,
      name = L[unlocName],
      class = class,
      show = (class == engClass or class == "ANY")
  })
end

-- It's important to have the english names available
-- to store settings with unique identifiers for
-- multi-language clients.

-- Warlock
AC_insertSelfBuff("Flameshadow", "WARLOCK") -- T4 proc
AC_insertSelfBuff("Shadowflame", "WARLOCK") -- T4 proc
AC_insertSelfBuff("Shadow Trance", "WARLOCK") -- Talent proc

-- Paladin
AC_insertSelfBuff("Divine Shield", "PALADIN") -- Skill

-- Warrior
AC_insertSelfBuff("Flurry", "WARRIOR") -- Talent proc
AC_insertSelfBuff("Berserker Rage", "WARRIOR") -- Skill
AC_insertSelfBuff("Retaliation", "WARRIOR") -- Skill
AC_insertSelfBuff("Death Whish", "WARRIOR") -- Skill
AC_insertSelfBuff("Shield Wall", "WARRIOR") -- Skill
AC_insertSelfBuff("Last Stand", "WARRIOR") -- Skill
AC_insertSelfBuff("Recklessness", "WARRIOR") -- Skill
AC_insertSelfBuff("Spell Reflection", "WARRIOR") -- Skill
AC_insertSelfBuff("Enrage", "WARRIOR") -- Talent proc

-- Hunter
AC_insertSelfBuff("The Beast Within", "HUNTER") -- Talent skill
AC_insertSelfBuff("Feign Death", "HUNTER") -- Skill
AC_insertSelfBuff("Rapid Fire", "HUNTER") -- Skill
AC_insertSelfBuff("Rapid Killing", "HUNTER") -- Talent
AC_insertSelfBuff("Quick Shots", "HUNTER") -- Talent

-- Misc
AC_insertSelfBuff("Clearcasting", "ANY") -- Talent proc (mage, druid, etc)
AC_insertSelfBuff("Power Word: Shield", "ANY") -- Priest skill
AC_insertSelfBuff("Prayer of Mending", "ANY") -- Priest skill
AC_insertSelfBuff("Blessing of Protection", "ANY") -- Paladin skill
AC_insertSelfBuff("Pain Suppression", "ANY") -- Priest Disci
AC_insertSelfBuff("Divine Protection", "ANY") -- Pala buff

-- Race
AC_insertSelfBuff("Berserking", "ANY") -- Race skill

-- Trinkets
AC_insertSelfBuff("Spell Power", "ANY") -- Spelldamage trinket
AC_insertSelfBuff("Arcane Energy", "ANY") -- Spelldamage trinket
AC_insertSelfBuff("Focused Power", "ANY") -- Spelldamage trinket
AC_insertSelfBuff("Crusade Aura", "ANY") -- Spelldamage trinket
AC_insertSelfBuff( "Ferocity", "ANY") -- Attackpower trinket
AC_insertSelfBuff("Heroism", "ANY") -- Attackpower trinket
AC_insertSelfBuff("Vigilance of the Colossus", "ANY") -- Heals on Block
AC_insertSelfBuff("Enlightenment", "ANY") -- ??
AC_insertSelfBuff("Rage of the Unraveller", "ANY") -- Attackpower trinket
AC_insertSelfBuff("Time's Favor", "ANY") -- Dodge rating

-- Heroic Trinkets
AC_insertSelfBuff("Gnome Ingenuity", "ANY") -- Increases block rating
AC_insertSelfBuff("Blessing of the Silver Crescent", "ANY") -- Spelldamage
AC_insertSelfBuff("Lust for Battle", "ANY") -- Attack power

-- Procs
AC_insertSelfBuff("Lightning Speed", "ANY") -- Mungo proc

--[ Spell-ready events ]-----------------------------------------------

local spellReadyEvents = {}

function AC_insertSpellReady (unlocName, class, texture)
  table.insert(spellReadyEvents, {
    id = unlocName,
    name = L[unlocName],
    class = class,
    show = (class == engClass or class == "ANY"),
    texture = texture,
  })
end

-- Warrior
AC_insertSpellReady("Overpower", "WARRIOR", iconOverpower)
AC_insertSpellReady("Execute", "WARRIOR", iconExecute)
AC_insertSpellReady("Revenge", "WARRIOR", iconRevenge)
AC_insertSpellReady("Victory Rush", "WARRIOR", iconVictoryRush)

-- Hunters
AC_insertSpellReady("Kill Command", "HUNTER", iconKillCommand)

--[ Addon default config ]---------------------------------------------

local defaults = {
  debug = false,
  animation = true,
  offsetyStandardBar = 300,
  offsetyNotices = 150,
  fadeOutOfCombat = true,
  fadeOutOfMeeleCombat = false,
  showKillingBlows = true,
  hideErrorFrame = false,
  selfBuffsEnabled = {},
  reactivesEnabled = {},
  userDefinedBuffs = {},
}

-- Init default settings for selfbuffs and reactives.
for i = 1, getn(selfBuffEvents) do
  defaults.selfBuffsEnabled[selfBuffEvents[i].id] = true
end

for i = 1, getn(spellReadyEvents) do
  defaults.reactivesEnabled[spellReadyEvents[i].id] = true
end

--[ Region pool functions ]--------------------------------------------

local Regions = {}

function Regions:request ()
  if getn(self) > 0 then
    return table.remove(self)
  else
    if ActionCombat.db.profile.debug then
      ActionCombat:Print("Creating region")
    end
    return AC_ANIM:CreateTexture()
  end
end

function Regions:release (region)
  region:Hide()
  table.insert(self, region)
end

--[ Animation sequences ]---------------------------------------------

local Sequences = {
  -- Animation sequence for selfbuff events
  SELFBUFF = {
    [1] = { ani = "POP_IN", len = aniFadeinLen },
    [2] = { ani = "POP_BACK", len = aniFadebackLen },
    [3] = { ani = "IDLE", len = nil },
    [4] = { ani = "DEACTIVATE", len = aniDeactivateLen }
  },

  -- Animaion sequence for reactive abilities
  REACTIVE = {
    [1] = { ani = "POP_IN", len = aniFadeinLen },
    [2] = { ani = "POP_BACK", len = aniFadebackLen },
    [3] = { ani = "IDLE", len = aniReactiveIdleLen },
    [4] = { ani = "DEACTIVATE", len = aniDeactivateLen }
  },

  -- Animaion sequence for notices
  NOTICE = {
    [1] = { ani = "NOTICE_FLASH", len = aniNoticeLen }
  },

  -- Animaion sequence for fading buffs
  BUFFFADE = {
    [1] = { ani = "SHRINK_IN", len = aniFadeinLen },
    [2] = { ani = "FADE_OUT", len = aniDeactivateLen }
  }
}

--[ Animation class / methods ]---------------------------------------

local Animation = {
  -- Age of the animation
  age = 0,
  -- Animation sequence stuff
  sequence = nil,
  current = 1,
  timer = 0,
  nohang = false,
  -- Position
  posx = 0,
  posy = 0,
  -- Name and texture
  texture = nil,
  name = nil,
  icons = nil,
}

-- Creates a new animation with the texture field set
function Animation:new (texture)
  local ani = {}
  ani.texture = texture
  ani.icons = {}

  setmetatable(ani, self)
  self.__index = self

  return ani
end

-- Sets up basic attributes: age, name, sequence, and region stuff
function Animation:setup (name, sequence)
  self.age = 0
  self.name = name
  self.sequence = sequence

  local icon = Regions:request()
  icon:SetTexCoord(0.1, 0.9, 0.1, 0.9)
  icon:SetHeight(0)
  icon:SetWidth(0)
  icon:SetTexture(self.texture)
  icon:Show()
  table.insert(self.icons, icon)
end

-- Moves all used icons to newx, newy
function Animation:moveto (newx, newy)
  self.posx = newx
  self.posy = newy
  for i = 1, #self.icons do
    self.icons[i]:ClearAllPoints()
    self.icons[i]:SetPoint("CENTER", "AC_ANIM", "CENTER", newx, newy)
  end
end

-- Releases all used icons to the region pool
function Animation:delete ()
  for i = #self.icons, 1, -1 do
    Regions:release(table.remove(self.icons))
  end
end

--[ The Scene -- Animations are managed here ]---------------------------

local Scene = {
  -- The standard window where nearly all animations appear
  winStandardBar = {
    offsetx = 0,
    offsety = 300,
    debugWidth = 500,
    debugHeight = 70,
    debugIcon = nil,
    anis = {}
  },

  -- The notification window, where notices (e.g. killing blows) appear
  winNotices = {
    offsetx = 0,
    offsety = 150,
    debugWidth = 180,
    debugHeight = 180,
    debugIcon = nil,
    anis = {}
  },

  -- Actively check if anis need to be realigned
  checkRealign = true,
  fadeOutOfCombat = nil,
  fadeOutOfMeeleCombat = nil,
  tempVisibleWindows = nil,
}

-- Initializes the scene
function Scene:init ()
  self:initDebugIcon(self.winStandardBar)
  self:initDebugIcon(self.winNotices)
end

-- Inits the debug windows (the animation bar outline in debug mode)
function Scene:initDebugIcon (win)
  win.debugIcon = Regions:request()
  win.debugIcon:SetWidth(win.debugWidth)
  win.debugIcon:SetHeight(win.debugHeight)
  win.debugIcon:SetAlpha(windowDebugAlphas[1])
  win.debugIcon:SetTexture(0, 1, 0)
  self:updateWindowDebugPos(win)
  win.debugIcon:Show()
end

-- Makes debug windows visible
function Scene:setWindowVisibility (value)
  debugIndex = value and 2 or 1
  self.winStandardBar.debugIcon:SetAlpha(windowDebugAlphas[debugIndex])
  self.winNotices.debugIcon:SetAlpha(windowDebugAlphas[debugIndex])
end

-- Sets new position for the debug windows (animation bar outline)
function Scene:updateWindowDebugPos (win)
  win.debugIcon:ClearAllPoints()
  win.debugIcon:SetPoint("CENTER", "AC_ANIM", "CENTER",
      win.offsetx, win.offsety)
end

-- Inserts an animation to the scene
function Scene:insertNewAni (name, texture, sequence)
  --ActionCombat:Print("Inserting new animation: "..name)

  -- Create new animation
  local animation = Animation:new(texture)
  animation:setup(name, sequence)

  -- Put animation in the right window
  if sequence == "SELFBUFF"
      or sequence == "REACTIVE"
      or sequance == "BUFFFADE" then

    local found = nil
    for i = 1, #self.winStandardBar.anis do
      if name == self.winStandardBar.anis[i].name then found = i end
    end

    if found == nil then
      -- Ani was not found. Append to the window.
      -- The fake thing is just for better icon placing.
      local fake = (#self.winStandardBar.anis == 0 and 0 or 1)
      table.insert(self.winStandardBar.anis, animation)
      animation:moveto(self:getIconPosition(self.winStandardBar,
          #self.winStandardBar.anis + fake, #self.winStandardBar.anis + fake))
    else
      -- Ani was found. Take old positonis and replace.
      foundAni = self.winStandardBar.anis[found]
      animation:moveto(foundAni.posx, foundAni.posy)
      foundAni:delete()
      self.winStandardBar.anis[found] = animation
    end
  elseif sequence == "NOTICE" then
    -- Insert notices by replacing previous animation, if any
    for i = 1, #self.winNotices.anis do self.winNotices.anis[i]:delete() end
    self.winNotices.anis = { animation }
    animation:moveto(self:getIconPosition(self.winNotices, 1))
  end

  self.checkRealign = true
  return animation
end

-- Inserts a Buff-Ani if the buff exists
function Scene:insertBuffIfExist (name)
  selfBuffAniRequest = name
  ActionCombat:PLAYER_AURAS_CHANGED()
end

-- Deactives an animation from the scene, this is done by setting
-- the nohang flag, and nil-length sequence parts are exited.
function Scene:deactivateAni (name)
  for i = 1, #self.winStandardBar.anis do
    if name == self.winStandardBar.anis[i].name then
      -- Deactivate the selfbuff by setting its current animation
      self.winStandardBar.anis[i].nohang = true
      break
    end
  end
end

-- Deactivates all animations
function Scene:deactivateAll ()
  for i = #self.winStandardBar.anis, 1, -1 do
    self.winStandardBar.anis[i].nohang = true
  end
end

-- Removes all animations
function Scene:removeAll ()
  for i = #self.winStandardBar.anis, 1, -1 do
    self.winStandardBar.anis[i]:delete()
  end

  for i = #self.winNotices.anis, 1, -1 do
    self.winNotices.anis[i]:delete()
  end

  self.winStandardBar.anis = {}
  self.winNotices.anis = {}
end

-- Helper function. Returns the desired position for an icon.
function Scene:getIconPosition (window, index, fakeAmount)
  local iconAmount = #window.anis
  if fakeAmount ~= nil then iconAmount = fakeAmount end

  local totalWidth = (iconAmount - 1) * (iconSpacing + iconSize)
  local iconPos = (index - 1) * (iconSpacing + iconSize)
  local destx = window.offsetx + iconPos - totalWidth / 2.0
  local desty = window.offsety
  return destx, desty
end

-- Realigns animations in the specified animation window
function Scene:realignWindowAnis (timediff, window)
  local iconAmount = #window.anis

  local somethingChanged = false
  for i = 1, iconAmount do
    local destx, desty = self:getIconPosition(window, i)

    -- Advance position with a speed of realignSpeed
    local newx = window.anis[i].posx
    if destx > window.anis[i].posx then
      newx = window.anis[i].posx + timediff * realignSpeed
      if newx > destx then newx = destx end
      somethingChanged = true
    elseif destx < window.anis[i].posx then
      newx = window.anis[i].posx - timediff * realignSpeed
      if newx < destx then newx = destx end
      somethingChanged = true
    end

    window.anis[i]:moveto(newx, desty)
  end

  return somethingChanged
end

-- Realigns animations in the starand animation bar
function Scene:realignAnimations (timediff)
  if not Scene.checkRealign then return end
  local sthChanged1 = self:realignWindowAnis(timediff, self.winStandardBar)
  local sthChanged2 = self:realignWindowAnis(timediff, self.winNotices)
  if not sthChanged1 and not sthChanged2 then Scene.checkRealign = false end
end

-- Advances sequence for a single animation. This is recursively called.
function Scene:advanceAnimation (ani, timediff)
  if ani.current > #Sequences[ani.sequence] then return end
  -- Advance timer for the active sequence part
  ani.timer = ani.timer + timediff
  -- Check if length of the sequence part has exceeded
  if Sequences[ani.sequence][ani.current].len == nil then
    -- If we're in a nil-length sequence part, we only need to care
    -- about the nohang flag. If set, advance part, set timer
    -- and recursively call with the same timediff
    if ani.nohang then
      ani.current = ani.current + 1
      ani.timer = 0
      self:advanceAnimation(ani, timediff)
    end
  elseif ani.timer > Sequences[ani.sequence][ani.current].len then
    -- If the sequence part is expired, advance part, readjust timer
    -- and recursively call (the new sequence part may also be expired).
    ani.timer = ani.timer - Sequences[ani.sequence][ani.current].len
    ani.current = ani.current + 1
    self:advanceAnimation(ani, 0)
  end
end

-- Advances all animations for a window and check for expiration
function Scene:advanceWindowAnimations (window, timediff)
  local anis = window.anis

  for i = #anis, 1, -1 do
    self:advanceAnimation(anis[i], timediff)
    if anis[i].current > #Sequences[anis[i].sequence] then
      -- Animation is expired
      --ActionCombat:Print("Animation deleted: "..anis[i].name
      --    ..", active: "..(#anis - 1));
      anis[i]:delete()
      table.remove(anis, i)
      self.checkRealign = true
    else
      -- This actually sets new positions, alphas, etc
      self:animate(anis[i])
    end
  end
end

-- Advances animations for all windows (listed here)
function Scene:advanceAllAnimations (timediff)
  self:advanceWindowAnimations(self.winStandardBar, timediff)
  self:advanceWindowAnimations(self.winNotices, timediff)
end

-- Animations
function Scene:animate (ani)
  local seqPart = Sequences[ani.sequence][ani.current]
  local timer = ani.timer
  local length = seqPart.len
  local progress = (length == nil or timer < 0) and 0 or timer / length

  local baseAlpha = aniBaseAlpha
  if (not inCombat and self.fadeOutOfCombat)
      or (not inMeeleCombat and self.fadeOutOfMeeleCombat) then
    baseAlpha = aniFadedBaseAlpha
  end

  if seqPart.ani == "POP_IN" then
    local size = iconSize * aniFadeInSizeFactor * progress
    ani.icons[1]:SetWidth(size)
    ani.icons[1]:SetHeight(size)
    ani.icons[1]:SetAlpha(baseAlpha)
  elseif seqPart.ani == "POP_BACK" then
    local startSize = iconSize * aniFadeInSizeFactor
    local size = progress * (iconSize - startSize) + startSize
    ani.icons[1]:SetWidth(size)
    ani.icons[1]:SetHeight(size)
    ani.icons[1]:SetAlpha(baseAlpha)
  elseif seqPart.ani == "IDLE" then
    local size = math.sin(-timer * 5) * 5 + iconSize
    ani.icons[1]:SetWidth(size)
    ani.icons[1]:SetHeight(size)
    ani.icons[1]:SetAlpha(baseAlpha)
  elseif seqPart.ani == "DEACTIVATE" then
    local size = iconSize - progress * iconSize
    local alpha = baseAlpha * (1.0 - progress)
    ani.icons[1]:SetWidth(size)
    ani.icons[1]:SetHeight(size)
    ani.icons[1]:SetAlpha(alpha)
  elseif seqPart.ani == "NOTICE_FLASH" then
    local size = iconSize * aniNoticeSizeFactor * progress
    local alpha = baseAlpha * (1.0 - progress)
    ani.icons[1]:SetWidth(size)
    ani.icons[1]:SetHeight(size)
    ani.icons[1]:SetAlpha(alpha)
  elseif seqPart.ani == "SHRINK_IN" then
  elseif seqPart.ani == "SHRINK_OUT" then
  end
end

------------------------------------------------------------------------

function ActionCombat:OnInitialize()
  options = {
    type = "group",
    args = {
      windowPos = {
        type = "group",
        name = L["MENU_WINDOW_POS_NAME"],
        desc = L["MENU_WINDOW_POS_DESC"],
        order = 10,
        args = {
          win1 = {
            type = "range",
            name = L["MENU_WINDOW_MAIN_NAME"],
            desc = L["MENU_WINDOW_POS_DESC"],
            order = 10,
            max = 500,
            min = -500,
            step = 5,
            get = function() return Scene.winStandardBar.offsety end,
            set = function(arg)
                Scene.winStandardBar.offsety = arg
                self.db.profile.offsetyStandardBar = arg
                Scene.tempVisibleWindows = windowDebugStayOnChange
                Scene:updateWindowDebugPos(Scene.winStandardBar)
                Scene:setWindowVisibility(true)
            end
          },
          win2 = {
            type = "range",
            name = L["MENU_WINDOW_NOTICE_NAME"],
            desc = L["MENU_WINDOW_POS_DESC"],
            order = 20,
            max = 500,
            min = -500,
            step = 5,
            get = function() return Scene.winNotices.offsety end,
            set = function(arg)
                Scene.winNotices.offsety = arg
                self.db.profile.offsetyNotices = arg
                Scene.tempVisibleWindows = windowDebugStayOnChange
                Scene:updateWindowDebugPos(Scene.winNotices)
                Scene:setWindowVisibility(true)
            end
          },
        }
      },
      enabled = {
        type = "toggle",
        name = L["MENU_ENABLED_NAME"],
        desc = L["MENU_ENABLED_DESC"],
        order = 20,
        get = function() return self.db.profile.animation end,
        set = function()
            self.db.profile.animation = not self.db.profile.animation
            if (self.db.profile.animation) then
              AC_ANIM:Show()
            else
              AC_ANIM:Hide()
            end
        end
      },
      fadeOutOfCombat = {
        type = "toggle",
        name = L["MENU_FADE_OUTOF_COMBAT_NAME"],
        desc = L["MENU_FADE_OUTOF_COMBAT_DESC"],
        usage = "",
        order = 21,
        get = function() return self.db.profile.fadeOutOfCombat end,
        set = function() self.db.profile.fadeOutOfCombat
            = not self.db.profile.fadeOutOfCombat
            Scene.fadeOutOfCombat = self.db.profile.fadeOutOfCombat
        end
      },
      fadeOutOfMeeleCombat = {
        type = "toggle",
        name = L["MENU_FADE_OUTOF_MEELE_COMBAT_NAME"],
        desc = L["MENU_FADE_OUTOF_MEELE_COMBAT_DESC"],
        usage = "",
        order = 22,
        get = function() return self.db.profile.fadeOutOfMeeleCombat end,
        set = function() self.db.profile.fadeOutOfMeeleCombat
            = not self.db.profile.fadeOutOfMeeleCombat
            Scene.fadeOutOfMeeleCombat = self.db.profile.fadeOutOfMeeleCombat
        end
      },
      debug = {
        type = "toggle",
        name = L["MENU_DEBUG_NAME"],
        desc = L["MENU_DEBUG_DESC"],
        usage = "",
        order = 30,
        get = function() return self.db.profile.debug end,
        set = function() self.db.profile.debug = not self.db.profile.debug end
      },
      --[[
      showbuffs = {
        type = "execute",
        name = L["MENU_SHOWBUFFS_NAME"],
        desc = L["MENU_SHOWBUFFS_DESC"],
        order = 40,
        func = function() self:showPlayerBuffs() end
      },
      ]]
      hideErrorFrame = {
        type = "toggle",
	name = L["MENU_HIDE_ERROR_FRAME_NAME"],
	desc = L["MENU_HIDE_ERROR_FRAME_DESC"],
	order = 45,
	set = function()
	  if UIErrorsFrame:IsVisible() then
            UIErrorsFrame:Hide()
	    self.db.profile.hideErrorFrame = true
	  else
	    UIErrorsFrame:Show()
	    self.db.profile.hideErrorFrame = false
	  end
	end,
	get = function() return not UIErrorsFrame:IsVisible() end
      },
      spacing1 = {
        type = "header",
        name = " ",
        order = 50
      },
      buffevents = {
        type = "group",
        order = 60,
        name = L["MENU_BUFFEVENTS_NAME"],
        desc = L["MENU_BUFFEVENTS_DESC"],
        args = {
        }
      },
      reactives = {
        type = "group",
        order = 70,
        name = L["MENU_REACTIVES_NAME"],
        desc = L["MENU_REACTIVES_DESC"],
        args = {
        }
      },
      miscEvents = {
        type = "group",
        order = 75,
        name = L["MENU_MISC_EVENTS_NAME"],
        desc = L["MENU_MISC_EVENTS_DESC"],
        args = {
	  showKillingBlows = {
            type = "toggle",
            name = L["MENU_SHOW_KILLING_BLOWS_NAME"],
            desc = L["MENU_SHOW_KILLING_BLOWS_DESC"],
            usage = "",
            order = 10,
            get = function() return self.db.profile.showKillingBlows end,
            set = function() self.db.profile.showKillingBlows
	        = not self.db.profile.showKillingBlows
            end
	  },
        }
      },
      userDefinedBuffs = {
        type = "group",
        order = 80,
        name = L["MENU_USER_BUFFS_NAME"],
        desc = L["MENU_USER_BUFFS_DESC"],
        args = {
          addBuff = {
            type = "text",
            order = 10,
            name = L["MENU_INSERT_USER_BUFF_NAME"],
            desc = L["MENU_INSERT_USER_BUFF_DESC"],
            usage = "",
            set = function(text) self:InsertUserDefinedBuff(text) end,
            get = function() return "" end
          },
          deleteBuff = {
            type = "group",
            order = 20,
            name = L["MENU_DELETE_USER_BUFF_NAME"],
            desc = L["MENU_DELETE_USER_BUFF_NAME"],
            args = {
            }
          },
        }
      },
    }
  }

  ActionCombat.options = options

  -- Setup FuBarPlugin
  self:SetIcon("Interface\\Icons\\INV_Sword_48")
  self:RegisterDB("ActionCombatDB", "ActionCombatDBPC")
  self:RegisterDefaults("profile", defaults)
  self:RegisterChatCommand("/actioncombat", "/ac", ActionCombat.options)

  -- Setup user defined buffs
  self:UpdateUserDefinedBuffs()

  -- Setup buff events / reactive events menu entries
  buffevents = options.args.buffevents.args
  reactives = options.args.reactives.args
  for i = 1, #allClasses do
    buffevents[allClasses[i]] = self:GetClassGroup(allClasses[i])
    reactives[allClasses[i]] = self:GetClassGroup(allClasses[i])
  end
  buffevents["ANY"] = self:GetClassGroup("ANY")
  reactives["ANY"] = self:GetClassGroup("ANY")

  for i = 1, #selfBuffEvents do
    table.insert(buffevents[selfBuffEvents[i].class].args,
        ActionCombat:GetBuffToggle(i))
  end
  for i = 1, #spellReadyEvents do
    table.insert(reactives[spellReadyEvents[i].class].args,
        ActionCombat:GetReactiveToggle(i))
  end

  for i = 1, #allClasses do
    if #buffevents[allClasses[i]].args == 0 then
      buffevents[allClasses[i]].disabled = true
    end
    if #reactives[allClasses[i]].args == 0 then
      reactives[allClasses[i]].disabled = true
    end
  end

  -- Register Menu
  self.OnMenuRequest = ActionCombat.options

  -- Cache some information
  Scene.fadeOutOfCombat = self.db.profile.fadeOutOfCombat
  Scene.fadeOutOfMeeleCombat = self.db.profile.fadeOutOfMeeleCombat
  Scene.winStandardBar.offsety = self.db.profile.offsetyStandardBar
  Scene.winNotices.offsety = self.db.profile.offsetyNotices

  -- Restore setting "Hide error frame"
  if self.db.profile.hideErrorFrame then
    UIErrorsFrame:Hide()
  end
end

-- When the addon is enabled...
function ActionCombat:OnEnable()
  -- Detect In-Fight / In-Meele-Combat
  self:RegisterEvent("PLAYER_REGEN_DISABLED")
  self:RegisterEvent("PLAYER_REGEN_ENABLED")
  self:RegisterEvent("PLAYER_ENTER_COMBAT")
  self:RegisterEvent("PLAYER_LEAVE_COMBAT")
  self:RegisterEvent("CHAT_MSG_COMBAT_HOSTILE_DEATH")
  -- To remove all animations
  self:RegisterEvent("PLAYER_DEAD")
  -- On buff gaines/fades, spell gets active, etc
  self:RegisterEvent("COMBAT_TEXT_UPDATE")
  -- Buff received/lost (texture is determined here)
  self:RegisterEvent("PLAYER_AURAS_CHANGED")

  if (self.db.profile.animation) then
    AC_ANIM:Show()
  end

  Scene:init()
end

-- When the addon is disabled
function ActionCombat:OnDisable()
end

------------------------------------------------------------------------

function ActionCombat:showPlayerBuffs()
  self:Print("You are currently buffed with:")
  for i = 1, 16 do
    local tex = GetPlayerBuffTexture(i)
    local name = GetPlayerBuffName(i)
    if tex ~= nil then
      self:Print(name..": "..tex)
    else break end
  end
end

function ActionCombat:GetUserBuffAction(buffName)
  return {
    type = "execute",
    name = buffName,
    desc = L["MENU_CLICK_TO_DELETE"],
    func = function()
        self:Print("Removing: "..buffName)
	self.db.profile.userDefinedBuffs[buffName] = nil
        self:UpdateUserDefinedBuffs()
    end
  }
end

function ActionCombat:GetUserBuffToggle(buffName)
  return {
    type = "toggle",
    name = buffName,
    desc = buffName,
    usage = "",
    set = function() self.db.profile.userDefinedBuffs[buffName]
        = not self.db.profile.userDefinedBuffs[buffName]
        if not self.db.profile.userDefinedBuffs[buffName] then
          Scene:deactivateAni(buffName)
        else
          Scene:insertBuffIfExist(buffName)
        end
    end,
    get = function() return self.db.profile.userDefinedBuffs[buffName] end
  }
end

function ActionCombat:GetClassGroup(className)
  local order
  if className == "ANY" then order = 10 else order = 20 end
  return {
      type = "group",
      name = L["CLASS_"..className],
      desc = L["CLASS_"..className],
      order = order,
      args = {
      }
  }
end

function ActionCombat:GetBuffToggle(i)
  local nameEN = selfBuffEvents[i].id
  return {
      type = "toggle",
      name = selfBuffEvents[i].name,
      desc = L["MENU_CLICK_TO_TOGGLE"],
      usage = "",
      order = i,
      disabled = not selfBuffEvents[i].show,
      get = function() return selfBuffEvents[i].show
          and self.db.profile.selfBuffsEnabled[nameEN]
      end,
      set = function()
        self.db.profile.selfBuffsEnabled[nameEN]
            = not self.db.profile.selfBuffsEnabled[nameEN]
        if not self.db.profile.selfBuffsEnabled[nameEN] then
          Scene:deactivateAni(selfBuffEvents[i].name)
        else
          Scene:insertBuffIfExist(selfBuffEvents[i].name)
        end
      end
  }
end

function ActionCombat:GetReactiveToggle(i)
  local nameEN = spellReadyEvents[i].id
  return {
      type = "toggle",
      name = spellReadyEvents[i].name,
      desc = L["MENU_CLICK_TO_TOGGLE"],
      order = i,
      disabled = not spellReadyEvents[i].show,
      get = function() return spellReadyEvents[i].show
          and self.db.profile.reactivesEnabled[nameEN]
      end,
      set = function() self.db.profile.reactivesEnabled[nameEN]
            = not self.db.profile.reactivesEnabled[nameEN]
      end
  }
end

function ActionCombat:InsertUserDefinedBuff (buffName)
  if buffName == "" then return end
  self:Print("Inserting Buff: "..buffName)
  self.db.profile.userDefinedBuffs[buffName] = true
  self:UpdateUserDefinedBuffs()
end


function ActionCombat:UpdateUserDefinedBuffs ()
  local userBuffs = ActionCombat.options.args.userDefinedBuffs.args

  -- Clear old list of buffs in the menu
  userBuffs.deleteBuff.args = {}
  for i = 1, #userBuffs do userBuffs[i] = nil end

  -- Add spacing
  if next(self.db.profile.userDefinedBuffs) ~= nil then
    table.insert(userBuffs, { type = "header", name = " ", order = 30 })
  end

  -- Re-add the buffs to the menu
  for buffName, buffActive in pairs(self.db.profile.userDefinedBuffs) do
    table.insert(userBuffs.deleteBuff.args, self:GetUserBuffAction(buffName))
    table.insert(userBuffs, self:GetUserBuffToggle(buffName))
  end
end

------------------------------------------------------------------------

function ActionCombat:PLAYER_AURAS_CHANGED()
  --if self.db.profile.debug then
  --  self:Print("PLAYER_AURAS_CHANGED!")
  --  self:showPlayerBuffs()
  --end

  if selfBuffAniRequest == nil then return end

  local found = false
  for id = 1, 16 do
    local name, rank = GetPlayerBuffName(id)
    local texture = GetPlayerBuffTexture(id)
    if name ~= nil and texture ~= nil then
      if name == selfBuffAniRequest then
        Scene:insertNewAni(selfBuffAniRequest, texture, "SELFBUFF")
        found = true
        selfBuffAniRequest = nil
        break
      end
    else break end
  end

  if not found then
    -- Buff already gone
    selfBuffAniRequest = nil
    if self.db.profile.debug then
      self:Print("Warning: Buff texture not found: "..selfBuffAniRequest)
    end
  end
end

function ActionCombat:CHAT_MSG_COMBAT_HOSTILE_DEATH(arg1)
  if self.db.profile.debug then
    self:Print(arg1)
  end

  -- FIXME: Maybe cache this translation to save CPU time?
  if self.db.profile.showKillingBlows
      and string.match(arg1, L["NOTICE_KILLING_BLOW"]) then
    if self.db.profile.debug then
      self:Print("KILLING BLOW!")
    end
    Scene:insertNewAni("Killing Blow!", iconKillingBlow, "NOTICE")
  end
end

function ActionCombat:PLAYER_REGEN_DISABLED()
  if self.db.profile.debug then
    self:Print("Entering Combat")
  end
  inCombat = true
end

function ActionCombat:PLAYER_REGEN_ENABLED()
  if self.db.profile.debug then
    self:Print("Leaving Combat")
  end
  inCombat = false
end

function ActionCombat:PLAYER_ENTER_COMBAT()
  if self.db.profile.debug then
    self:Print("Entering melee combat")
  end
  inMeeleCombat = true
end

function ActionCombat:PLAYER_LEAVE_COMBAT()
  if self.db.profile.debug then
    self:Print("Leaving melee combat")
  end
  inMeeleCombat = false
end

function ActionCombat:PLAYER_DEAD()
  if self.db.profile.debug then
    self:Print("Player died!")
  end
  Scene:removeAll()
end

function ActionCombat:COMBAT_TEXT_UPDATE(arg1, arg2, arg3)
  if self.db.profile.debug then
    self:Print("COMBAT_TEXT_UPDATE")
    if arg1 ~= nil then
      self:Print("  arg1: "..arg1)
    end
    if arg2 ~= nil then
      self:Print("  arg2: "..arg2)
    end
    if arg3 ~= nil then
      self:Print("  arg3: "..arg3)
    end
  end

  -- If you gain a buff
  if arg1 == "SPELL_AURA_START" then
    -- Check if this buff is registered
    for i = 1, #selfBuffEvents do
      if selfBuffEvents[i].show and arg2 == selfBuffEvents[i].name and
           self.db.profile.selfBuffsEnabled[selfBuffEvents[i].id] then
        -- Request the animation. Request is handled by PLAYER_AURAS_CHANGED.
        selfBuffAniRequest = arg2
        break
      end
    end

    if selfBuffAniRequest == nil then
      for buffName, buffActive in pairs(self.db.profile.userDefinedBuffs) do
        if buffActive and arg2 == buffName then selfBuffAniRequest = arg2 end
      end
    end

  -- If you loose a buff
  elseif arg1 == "SPELL_AURA_END" then
    Scene:deactivateAni(arg2)

  -- If a spell gets ready
  elseif arg1 == "SPELL_ACTIVE" then
    for i = 1, getn(spellReadyEvents) do
      if spellReadyEvents[i].show and spellReadyEvents[i].name == arg2 and
          self.db.profile.reactivesEnabled[spellReadyEvents[i].id] then
        Scene:insertNewAni(spellReadyEvents[i].name,
            spellReadyEvents[i].texture, "REACTIVE")
      end
    end
  end
end

------------------------------------------------------------------------

function ActionCombat:UpdateAnimation (elapsed)
  -- Advance global timer
  timeleft = timeleft + elapsed

  -- Animations are advanced first. This may cause them to expire.
  Scene:advanceAllAnimations(elapsed)

  -- Animations are realigned, expired animations are considerd.
  Scene:realignAnimations(elapsed)

  -- Check if the debug regions for windows need to be hidden.
  if Scene.tempVisibleWindows ~= nil then
    Scene.tempVisibleWindows = Scene.tempVisibleWindows - elapsed
    if Scene.tempVisibleWindows < 0 then
      Scene.tempVisibleWindows = nil
      Scene:setWindowVisibility(false)
    end
  end
end
