--[[

  An addon to keep the number of soul shards under control. It's useful because with improved drain
  soul I get a % of total mana back but don't want all the shards all the time. Will not delete
  shards from shard bags, only ordinary ones. 

  combatwombat created February 2007

  Syntax: /AKSShardMax max n

  Default is keep all shards

  Updated 29Apr08 Handle warlock females in German locale

  Updated 29Mar08 Updated toc number, use new API call for shard bag detection

  Updated 23Feb08 Added a delete command

  Updated 12Jun07 Shards are now deleted when changing the max setting to below the
  number of shards held

  Updated 05Jul07 DisableAddon changed to DisableAddOn
  
  Updated 22Jul07 Added German localisation (thanks Blixa on curse-gaming) and new parameters 
  spare and min

  Updated 26Jul07 Removed check for max being 0 in the delete routine, tweaked German
  localisation slightly

  Updated 28Sep07 Addon now properly disables itself if called for a player who is
  not a warlock

]]


local L = AceLibrary("AceLocale-2.2"):new("AKSShardMax")
local Time = GetTime()
local PlayerName 
local TotalShards = 0 
local TotalShards_old = 0 
local enabled = true
local opts = false

AKSShardMax = AceLibrary("AceAddon-2.0"):new("AceConsole-2.0", "AceEvent-2.0", "AceDB-2.0", "AceModuleCore-2.0")
AKSShardMax:RegisterDB("AKSShardMaxDB", "AKSShardMaxDBPC")
AKSShardMax:RegisterDefaults("char", {
  spare = 0,
  min = 1,
  msg = true,
} )

function AKSShardMax:OnInitialize()
end

function AKSShardMax:OnEnable()
  if UnitClass("Player") ~= L["Warlock"] and UnitClass("Player") ~= L["Warlock_Female"] then
    self:Print(L["You are not a warlock"])
    DisableAddOn("AKSShardMax")
    enabled = false
  else
    if not opts then
      opts = { 
        type='group',
        args = {
          max = {
            type = 'range',
            name = L["Max Shards"],
            desc = L["Sets the number of shards to keep"],
            usage = L["max N (0 means all shards)"],
            min = 0,
            max = 64,
            get = function() return self.db.char.max end,
            set = function(n) self.db.char.max = n self:ManageShards() end,
            error = L["Value outside the range 0..64"],
          },
          spare = {
            type = 'range',
            name = L["Spare slots"],
            desc = L["Auto-delete shards to keep this number of slots free. 0 to disable"],
            usage = L["spare N (0 means disable)"],
            min = 0,
            max = 64,
            get = function() return self.db.char.spare end,
            set = function(n) self.db.char.spare = n self:ManageShards() end,
            error = L["Value outside the range 0..64"],
          },
          min = {
            type = 'range',
            name = L["Min shards"],
            desc = L["Sets the minimum number of shards to keep"],
            usage = L["min N"],
            min = 0,
            max = 64,
            get = function() return self.db.char.min end,
            set = function(n) self.db.char.min = n self:ManageShards() end,
            error = L["Value outside the range 0..64"],
          },
          delete = {
            type = 'text',
            name = L["Delete shards"],
            desc = L["Instantly deletes the number of shards specified"],
            usage = L["N"],
            get = false,
            set = 
              function(n)
                if not n or string.find(n, "[^0-9]") or n == "" or tonumber(n) < 1 or tonumber(n) > 64 then
                  self:Print(L["Value outside the range 1..64"])
                else
                  self:AKSShardMaxDeleteShard(tonumber(n)) 
                end
              end,
          },
          msg = {
            type = 'toggle',
            name = L["Toggle Message"],
            desc = L["Turns on and off a message when a shard is deleted"],
            get = function() return self.db.char.msg end,
            set = function(v) self.db.char.msg = v end,
          },
        },
      }  
      self:RegisterChatCommand(L["Slash-Commands"], opts)
    end
    self:Print(L["Monitoring shards"])
    self:RegisterEvent("UNIT_INVENTORY_CHANGED")
    self:RegisterEvent("PLAYER_LOGGED_IN")
    if not self.db.char.max then
      self:Print(L["Initialised shardcount to zero"])
      self.db.char.max = 0
    end
  end
end

function AKSShardMax:UNIT_INVENTORY_CHANGED()
  if enabled then
    self:ManageShards()
  end
end

----------------------------------------------------------------------
-- called when our inventory changes
----------------------------------------------------------------------
function AKSShardMax:ManageShards()
  TotalShards = tonumber(self:AKSShardMaxCountShards())
  local max = self.db.char.max
  local min = self.db.char.min
  local DesiredFreeSlots = self.db.char.spare
  local ActualFreeSlots = self:AKSShardMaxCountFreeSlots()
  if not max then 
    max = 0 
  end
  if not min then
    min = 0
  end
  --------------------------------------------------------------------
  -- the time check is a bit unpleasant but I was finding that 
  -- this routine would get called twice. once when i gained a
  -- shard and once when I deleted a shard. Note to self - there
  -- has to be a more elegant way of handling this
  --------------------------------------------------------------------
  if GetTime() - Time >= 3 then
    if ActualFreeSlots < DesiredFreeSlots and DesiredFreeSlots > 0 then
      local NumToDelete = DesiredFreeSlots - ActualFreeSlots
      -- make sure we keep the minimum number of shards
      if TotalShards - NumToDelete < min then
        NumToDelete = TotalShards - min
      end
      if NumToDelete > 0 then
        self:AKSShardMaxDeleteShard(NumToDelete) 
      end
    end
    -- count how many shards we're left with and compare to the max
    -- value
    TotalShards = tonumber(self:AKSShardMaxCountShards())
    if TotalShards > max and max > 0 then
      self:AKSShardMaxDeleteShard(TotalShards - max) 
    end
  end
  Time = GetTime()
end

function AKSShardMax:PLAYER_LOGGED_IN()
  Time = GetTime()
end  

----------------------------------------------------------------------
-- count how many slots we have free in normal bags, ignore shard bags
----------------------------------------------------------------------
function AKSShardMax:AKSShardMaxCountFreeSlots() 
  local bag, slot, text
  local count = 0
  for bag = 0, NUM_BAG_FRAMES do
    if not self:AKSShardMaxIsShardBag(bag) then
      for slot = 1, GetContainerNumSlots(bag) do
        text = GetContainerItemLink(bag, slot);
        if not text then
         count = count + 1
        end
      end
    end
  end
  return count
end

----------------------------------------------------------------------
-- count how many shards we have in our inventory
----------------------------------------------------------------------
function AKSShardMax:AKSShardMaxCountShards() 
  local bag
  local slot
  local text
  local count = 0
   for bag = 0, NUM_BAG_FRAMES do
     for slot = 1, GetContainerNumSlots(bag) do
       text = GetContainerItemLink(bag, slot);
       if text and string.find(text, "%[" .. L["Soul Shard"] .. "%]") then
         count = count + 1
       end
    end
  end
  return count
end

----------------------------------------------------------------------
-- returns true if a bag is a shard bag, false otherwise
----------------------------------------------------------------------
function AKSShardMax:AKSShardMaxIsShardBag(bag)    
  local shardbag = false
  if bag > 0 then -- ignore backpack
    local bagLink = GetInventoryItemLink("player", ContainerIDToInventoryID(bag))
    local bagType = GetItemFamily(bagLink)
    if not bagType then -- bag slot is empty most probably
      return true
    else
      shardbag = (bagType == 4)
    end
  end
  return shardbag
end

----------------------------------------------------------------------
-- deletes shards from non-shard bags
----------------------------------------------------------------------
function AKSShardMax:AKSShardMaxDeleteShard(HowmanyToDelete) 
  local done = 0
  local bag, slot = 0
  local text
  local t = HowmanyToDelete
  if t <= 0 then
    return
  end
  for bag = 0, NUM_BAG_FRAMES do
    if not self:AKSShardMaxIsShardBag(bag) then
      for slot = 1, GetContainerNumSlots(bag) do
        local text = GetContainerItemLink(bag, slot)
        if text and string.find(text, "%[" .. L["Soul Shard"] .. "%]") then
          PickupContainerItem(bag, slot)
          DeleteCursorItem()
          if self.db.char.msg then
            self:Print(L["Deleted a shard at bag:slot"] .. " " .. bag .. ":" .. slot)
          end
          t = t - 1
          if t <= 0 then
            Time = GetTime()
            if self.db.char.msg then
              PlaySound("TellMessage")
            end
            return
          end
        end
      end
    end
  end
end

